package gr.ntua.ece.cslab.modissense.client;

import gr.ntua.ece.cslab.modissense.containers.POI;
import gr.ntua.ece.cslab.modissense.containers.UserPOIQueryArguments;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;

import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

public class UserPOIQuery {

	public static final String 	TABLE_NAME = "TestingTable";

	private UserPOIQueryArguments arguments;
	private HTable table;
	private Long[] cuts;
	private long time;

	private List<POI> results;

	public UserPOIQuery() {
		try {
			this.table = new HTable(HBaseConfiguration.create(), TABLE_NAME);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}


	public UserPOIQueryArguments getArguments() {
		return arguments;
	}


	public void setArguments(UserPOIQueryArguments arguments) {
		this.arguments = arguments;
	}


	public void initialize() {
		this.time = System.currentTimeMillis();
		this.estimateCuts();
	}

	private void estimateCuts() {
		Set<Long> cuts = new TreeSet<Long>();
		Pair<byte[][], byte[][]> pair;
		try {
			pair = table.getStartEndKeys();
			for(int i=0;i<pair.getFirst().length;i++){
				byte[] start= pair.getFirst()[i], stop = pair.getSecond()[i];
				if(start.length>0)
					cuts.add(Bytes.toLong(start));
				if(stop.length>0)
					cuts.add(Bytes.toLong(stop));
			}
			this.cuts = new Long[cuts.size()];
			cuts.toArray(this.cuts);
		} catch (IOException e) {
			e.printStackTrace();
		}

	}

	public void execute() throws InterruptedException {
		List<UserPOIRegionThread> threads = new LinkedList<UserPOIRegionThread>();
		for(int i=0;i<this.cuts.length+1;i++){
			List<Long> users = this.getUsersIntoPartition(i);
			if(users!=null){
				UserPOIQueryArguments args = this.arguments.clone();
				args.setUserIds(users);
				threads.add(new UserPOIRegionThread(table, args));
			}
		}
		for(Thread t: threads)
			t.start();

		for(Thread t: threads)
			t.join();

		// centralized approach
		HashMap<Long, List<POI>> results = new HashMap<Long, List<POI>>();

		for(UserPOIRegionThread t:threads){
			for(POI p:t.getResults().getPOIs()){
				if(!results.containsKey(p.getId()))
					results.put(p.getId(), new LinkedList<POI>());
				results.get(p.getId()).add(p);
			}
		}
		
		this.results = this.getSortedResults(results); 
	}

	public List<POI> getSortedResults(Map<Long, List<POI>> pois) {
		LinkedList<POI> results = new LinkedList<POI>();
		for(Entry<Long, List<POI>> e :pois.entrySet()) {
			double sum=0.0;
			for(POI p:e.getValue())
				sum+=p.getScore();
			POI p = new  POI();
			p.setId(e.getValue().get(0).getId());
			p.setKeywords(e.getValue().get(0).getKeywords());
			p.setName(e.getValue().get(0).getName());
			p.setX(e.getValue().get(0).getX());
			p.setY(e.getValue().get(0).getY());
			p.setScore(sum/e.getValue().size());
			results.add(p);
		}
		Collections.sort(results, 
				new Comparator<POI>() {
			public int compare(POI o1, POI o2) {
				if(o1.getScore()>o2.getScore())
					return -1;
				else if(o1.getScore()<o2.getScore())
					return 1;
				else
					return 0;
			};
		});
		return results;
	}

	public List<Long> getUsersIntoPartition(int partition) {
		if(partition>this.cuts.length)
			return null;


		long from = (partition==0?0l:this.cuts[partition-1]), to = (partition<this.cuts.length?this.cuts[partition]:Long.MAX_VALUE);
		int leftIndex=0, rightIndex;
		while(leftIndex < arguments.getUserIds().size() && arguments.getUserIds().get(leftIndex) < from )
			leftIndex++;
		if(leftIndex>=this.arguments.getUserIds().size())
			return null;
		rightIndex = leftIndex;
		while(rightIndex < this.arguments.getUserIds().size() && this.arguments.getUserIds().get(rightIndex) < to )
			rightIndex++;
		if(rightIndex<=leftIndex)
			return null;
		return this.arguments.getUserIds().subList(leftIndex, rightIndex);
	}

	public void finish() {
		this.time = System.currentTimeMillis()-time;
		try {
			this.table.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	public List<POI> getResults() {
		return this.results;
	}
	
	public long getTime(){
		return this.time;
	}

	public static void main(String[] args) throws InterruptedException {
		UserPOIQuery query = new UserPOIQuery();
		int usersCount = new Integer(args[0]);
		Random rand = new Random();
		List<Long> userList = new ArrayList<Long>();
		for(int i=0;i<usersCount;i++){
			userList.add(Math.abs(rand.nextLong()%50000 + 1));
		}

		Collections.sort(userList);

		UserPOIQueryArguments arguments = new UserPOIQueryArguments();

		arguments.setUserIds(userList);
		arguments.setxFrom(0.0);
		arguments.setyFrom(0.0);
		arguments.setxTo(10.0);
		arguments.setyTo(10.0);
		arguments.addKeyword("bar");
		//		arguments.addKeyword("");


		query.setArguments(arguments);

		query.initialize();
		query.execute();
		query.finish();
		System.out.println(query.time);
	}

}
